/** @file   war.h
 * @brief   Declaration of War - class.
 * @version $Revision: 1.7 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_WAR_H
#define H_WWW_WAR_H

#include <vector>
#include <string>

#include "gameobject.h"
#include "bullettable.h"
#include "player.h"
#include "door.h"
#include "bonusobject.h"
#include "weather.h"
#include "objectivenote.h"
#include "redrawqueue.h"
#include "objectstorage.h"
#include "hud.h"
#include "level.h"
#include "MMp3PlayerProvider.h"

namespace WeWantWar {

// Forward declarations
class LightTable;
class ComboMeter;

/** @class  War
 * @brief   Implements the action part of the game.
 * @author  Tomi Lamminsaari
 *
 * War - class manages the actionpart of the game. When new level is about to
 * being played, you must call the <code>prepareLevel(...)</code> - method.
 * It loads the map, creates the game objects and initializes all the
 * datastructures needed in the game.
 * 
 * After the initialization you must call the <code>play()</code> method to
 * start the gameplay. The excecution returns from this method after the
 * gameplay has been finished.
 * 
 * Then you should use the <code>getEndReason()</code> - method to find out
 * why the gameplay ended.
 * 
 * And finally you should call <code>destroyLevel</code> - method to cleanup
 * the datastructures allocated for the game.
 *
 */
class War : public MMp3PlayerProvider
{
public:

  /** From which index the Player-object can be found from object list */
  static const int PLAYER_INDEX = 0;
  
  /** Default value for the 'm_quitNow'-member. */
  static const int QUIT_NOT_YET = 0;
  /** Returned from 'getEndReason()'-method when player died. */
  static const int QUIT_PLAYER_DIED = 1;
  /** Returned from 'getEndReason()'-method when player quit by pressing esc. */
  static const int QUIT_ESC = 2;
  /** Returned from 'getEndReason()'-method when player played the level through. */
  static const int QUIT_LEVEL_COMPLETE = 3;
  /** Returns from  'getEndReason()' - method when objective was failed. */
  static const int QUIT_OBJECTIVE_FAILED = 4;
  
  static const int SECTOR_COLUMNS = 8;
  static const int SECTOR_ROWS = 8;

  /** Constructs new actionpart.
   */
  War();
  
  /** Destructor
   */
  ~War();
  


  /** Initializes new level for playing. This method must be called before
   * using calling the 'run()'-method.
   * @param     levelNum            Levelnumber
   * @return    0 if initialization was ok. Negative number if it failed.
   */
  int prepareLevel(int levelNum);
  
  /** Starts the gameplay. Before calling this, you must call 'prepareLevel()'-
   * method.
   */
  void play();
  
  /** Destroys the level and releases the resources.
   */
  void destroyLevel();
  


  /** Returns the reason the gameplay ended.
   * @return    War::QUIT_PLAYER_DIED, War::QUIT_ESC or War::QUIT_LEVEL_COMPLETE
   */
  int getEndReason() const;
  
  
  
  /** From MMp3PlayerProvider.
   */
  virtual void setMp3Volume( int aVolume );
  
  /** From MMp3PlayerProvider.
   */
  virtual void playMp3File( int aSongIndex );
  
  /** From MMp3PlayerProvider.
   */
  virtual int playbackVolume() const;

protected:

  


  /** The possible menupages */
  enum {
    /** The game itself is being played. */
    PAGE_GAME,
    /** The Game-menu. THe game itself is paused. */
    PAGE_PAUSEMENU,
    /** Menupage shown when player dies */
    PAGE_PLAYERDIED,
    /** Page that is shown when player has finished the level */
    PAGE_LEVELCOMPLETE,
    /** There's an infopage visible */
    PAGE_WEAPONGUIDE,
    /** Level fade in page. */
    PAGE_LEVELFADEIN,
    /** Game Over page. */
    PAGE_GAMEOVERSCREEN
  };
  
  
  /** This should be called when preparing new level for playing. This inits
   * the Player-object and Alien-object.
   */
  void createObjects();
  
  /** Creates the Alien- and Bonusobjects from the data stored in Map-files. */
  void createAlienObjects();
  
  /** If the GameObject is placed too close to the non-passable map block, this
   * method moves it to a coordinate where it's not too close to the walls.
   * @param     pO                Pointer to GameObject
   */
  void translateToFreeBlocks( GameObject* pO );
  
  /** The main update-method of the game. Based on current menupage, this
   * updates either the gameplay or menus.
   */
  void update();
  
  /** The method that updates the game-part.
   */
  void updateGame();
  
  /** Updates both, the Pause and Player died - menus
   */
  void updatePauseMenu();
  
  /** Updates the level fade in animation.
   */
  void updateLevelFadeIn();
  
  /** Updates the game over page.
   */
  void updateGameOverScreen();
  
  /** Checks the collisions between the gameobjects. */
  void checkGameObjectCollisions();
  
  /** Checks the collisions between the vehicle the player is driving and
   * GameObjects
   */
  void checkVehicleGameObjectCollisions();
  
  /** Checks the collisions between the flying bullets and GameObjects. */
  void checkBulletCollisions();
  
  
  /** Check if there's a collision between the given bullet and given
   * GameObject. If there is, this method launches the bullethit animation,
   * reduces the health-levels and makes sound.
   * @param     pB                Pointer to Bullet
   * @param     pO                pointer to GameObject
   * @return    <code>true</code> if the bullet should be removed from the
   *            bullet table.
   */
  bool checkBulletGameObjectCollision( Bullet* pB, GameObject* pO );
  
  /** Handles the situation where bullet hits a GameObject.
   * @param     pB                Pointer to Bullet.
   * @param     pO                Pointer to GameObject the bullet hits.
   * @return    'true' if the Bullet should be removed from the bullet table.
   */
  bool handleBulletHit( Bullet* pB, GameObject* pO );
  
  /** Checks the collisions between player and light fences
   */
  void checkPlayerLightfenceCollisions();
  
  /** Updates the HUD
   */
  void updateHud();
  
  /** Redraws everything. Based on current menupage, this method redraws
   * the gameplay or menus
   */
  void redraw();
  
  /** Redraws the gamepart.
   */
  void redrawGame();
  
  /** Redraws the pausemenu, player died-menu and level-complete menu
   */
  void redrawPauseMenu();
  
  /** Redraws the level fsade in effect.
   */
  void redrawLevelFadeIn();
  
  /** Redraws the game over screen.
   */
  void redrawGameOverScreen();
  
  /** Redraws the Map.
   * @param     l                 Which layer. 0 = background of layer 0,
   *                              1 = foreground of layer 0
   */
  void drawMap( int l );
  
  /** Draws the bonusobjects
   */
  void drawBonuses();
  
  /** Shows the story of this level. This method will be exited when user
   * presses a key.
   */
  void showLevelStory();
  
  /** Shows the game credits when the last level has been played through.
   */
  void showCredits();
  
  /** Shows some debug-information. This method will be exited when user
   * presses a key.
   */
  void showDebugInfo();
  
  /** This method waits until the user releases the key.
   * @param     scan              Scancode of the key we're waiting.
   */
  void whileKey(int scan);
  
  /** Shows the keyboardinfo.
   */
  void showKeyInfo();
  
  /** Checks if there are vehicles that the player could capture. If there are
   * this method makes the player to capture it.
   */
  void findVehicleToCapture();

  /** Grants the cheatmode bonuses to the player.
   */
  void grantCheatBonuses();
  
  /** Creates the lighting models.
   */
  void createLightModels();

  ///
  /// Members
  /// =======

  /** This indicates when the mainloop should exit. Default value will be
   * QUIT_NOT_YET
   */
  int  m_quitNow;

  
  /** A vector that holds the MP3-files we use in this level. */
  std::vector<eng2d::MP3File*> m_playlist;
  
  
  /** The weather-controller object. */
  Weather*    m_pWeather;
  
  /** The active menupage. */
  int         m_menupage;
  
  /** The selected menuitem in pause menu */
  int         m_selectedMenuItem;
  
  /** Scancode of the previus key */
  int         m_lastKey;
  
  /** Should we show the keyboard commands during the level startup */
  bool        m_showKeyInfo;
  
  /** Tells if the game is paused */
  bool        m_paused;
  
  /** Once negative, it's possible to go to PAUSE menu */
  int         m_pauseMenuDelay;
  
  /** The objective note */
  ObjectiveNote*  m_pObjectives;
  
  /** The redraw queue */
  RedrawQueue*    m_pRedrawQueue;
  
  /** Bitmap for pause-menu */
  BITMAP* m_pPauseBitmap;
  
  /** Bitmap for YOU'RE DEAD title */
  BITMAP* m_pDeadTitle;
  
  /** Bitmap for LEVEL COMPLETE - title */
  BITMAP* m_pLevelCompleteTitle;
  
  /** Pointer to the ObjectStorage - object that manages the GameObjects. */
  ObjectStorage* m_pObjManager;
  
  /** Pointer to LightTable object that manages the lights. */
  LightTable* m_lightTable;
    
  /** Pointer to HUD - object that draws the hud */
  Hud* m_pHud;
  
  /** Pointer to level. */
  Level* m_pLevel;
  
  /** Number of the Objective we're currently playing. */
  int m_currentObjective;
  
  /** The level fade in animation */
  int m_currentScanLine;
  
  /** The level area is divided in 64 sectors. This member has 64 vectors
   * and each of these vectors has pointer to the GameObjects that are inside
   * the partical sector.
   */
  std::vector< std::vector<GameObject*> > m_sectorVector;
  
  /** Set true when bullettime-button is pressed */
  bool m_bulletTimeButton;
  
  /** Pointer to ComboMeter object */
  ComboMeter* m_comboMeter;
  
  /** Current level number. */
  int m_levelNumber;
  
  /** Counter that keeps track how long the game over screen have been shown. */
  int m_gameOverCounter;
  
private:

  War(const War& rO);
  War& operator = (const War& rO);
};

};  // end of namespace


#endif

/**
 * Version history
 * ===============
 * $Log: war.h,v $
 * Revision 1.7  2006/08/13 21:05:54  lamminsa
 * Game Over screen added.
 *
 * Revision 1.6  2006/05/30 14:56:13  lamminsa
 * showCredits() method and m_levelNumber member added.
 *
 * Revision 1.5  2006/05/15 19:22:12  lamminsa
 * Implements the MMp3PlayerProvider interface.
 *
 * Revision 1.4  2006/04/03 20:56:26  lamminsa
 * Combo memeter added.
 *
 * Revision 1.3  2006/02/26 07:39:22  lamminsa
 * no message
 *
 * Revision 1.2  2006/02/11 23:41:56  lamminsa
 * LightTable object added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:19+02  lamminsa
 * Initial revision
 *
 */
 
